/* Pro Timer Dashboard — app.js
   Features:
   - Interval (warmup/work/rest/cooldown cycles)
   - Stopwatch (big display & laps)
   - Countdown (minutes:seconds)
   - Progress bars, sound + vibration, sound test, theme persistence
   - Bottom nav switching (SPA-style)
*/

/* -------------------------
   Helpers & Element refs
   ------------------------- */
const beep = document.getElementById('beepSound');
const soundTestBtn = document.getElementById('soundTestBtn');
const soundToggleBtn = document.getElementById('soundToggle');
const soundToggleLabel = document.getElementById('soundToggleLabel');
const themeToggle = document.getElementById('themeToggle');

// nav buttons (bottom)
const navBtns = document.querySelectorAll('.nav-btn');

// Sections
const sections = {
  interval: document.getElementById('intervalSection'),
  stopwatch: document.getElementById('stopwatchSection'),
  countdown: document.getElementById('countdownSection')
};

/* -------------------------
   THEME (persist)
   ------------------------- */
const THEME_KEY = 'pro_timer_theme';
function initTheme() {
  const saved = localStorage.getItem(THEME_KEY);
  const prefersDark = window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches;
  if (saved === 'dark' || (!saved && prefersDark)) {
    document.documentElement.classList.add('dark');
  } else {
    document.documentElement.classList.remove('dark');
  }
  updateThemeIcon();
}
function toggleTheme() {
  document.documentElement.classList.toggle('dark');
  const isDark = document.documentElement.classList.contains('dark');
  localStorage.setItem(THEME_KEY, isDark ? 'dark' : 'light');
  updateThemeIcon();
}
function updateThemeIcon() {
  const isDark = document.documentElement.classList.contains('dark');
  themeToggle.innerHTML = isDark
    ? '<svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M21 12.79A9 9 0 1111.21 3 7 7 0 0021 12.79z"/></svg>'
    : '<svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="4"/><path d="M12 2v2"/><path d="M12 20v2"/><path d="M4.93 4.93l1.41 1.41"/><path d="M17.66 17.66l1.41 1.41"/></svg>';
}
themeToggle.addEventListener('click', toggleTheme);

/* -------------------------
   SOUND CONTROL
   ------------------------- */
let SOUND_ENABLED = (localStorage.getItem('pro_timer_sound') !== 'off'); // default on
function updateSoundToggleUI() {
  soundToggleLabel.textContent = SOUND_ENABLED ? 'Sound On' : 'Sound Off';
}
soundToggleBtn.addEventListener('click', () => {
  SOUND_ENABLED = !SOUND_ENABLED;
  localStorage.setItem('pro_timer_sound', SOUND_ENABLED ? 'on' : 'off');
  updateSoundToggleUI();
});

// Sound test: play beep + vibrate (if available)
soundTestBtn.addEventListener('click', async () => {
  // Try to play sound (some browsers require user gesture which we have)
  try {
    if (SOUND_ENABLED) {
      await beep.play();
    }
    if (navigator.vibrate) navigator.vibrate([150, 80, 150]);
  } catch (e) {
    console.warn('Sound play failed', e);
    alert('Sound could not play — check browser autoplay permissions.');
  }
});
updateSoundToggleUI();

/* -------------------------
   NAVIGATION (SPA-like)
   ------------------------- */
function showSection(key) {
  // hide all
  Object.values(sections).forEach(s => s.classList.add('hidden'));
  // remove active from nav
  navBtns.forEach(b => b.classList.remove('active'));
  // show selected
  if (key === 'interval') {
    sections.interval.classList.remove('hidden');
    document.querySelector('.nav-btn[data-target="intervalSection"]').classList.add('active');
  } else if (key === 'stopwatch') {
    sections.stopwatch.classList.remove('hidden');
    document.querySelector('.nav-btn[data-target="stopwatchSection"]').classList.add('active');
  } else if (key === 'countdown') {
    sections.countdown.classList.remove('hidden');
    document.querySelector('.nav-btn[data-target="countdownSection"]').classList.add('active');
  }
}
navBtns.forEach(btn => {
  btn.addEventListener('click', () => {
    const target = btn.dataset.target;
    // map id to key
    if (target === 'intervalSection') showSection('interval');
    if (target === 'stopwatchSection') showSection('stopwatch');
    if (target === 'countdownSection') showSection('countdown');
  });
});

/* -------------------------
   INTERVAL TIMER
   ------------------------- */
const intervalTimeEl = document.getElementById('intervalTime');
const intervalProgressEl = document.getElementById('intervalProgress');
const intervalStatusEl = document.getElementById('intervalStatus');

const warmupInput = document.getElementById('warmup');
const workInput = document.getElementById('work');
const restInput = document.getElementById('rest');
const cyclesInput = document.getElementById('cycles');
const cooldownInput = document.getElementById('cooldown');

const intervalStartBtn = document.getElementById('intervalStart');
const intervalPauseBtn = document.getElementById('intervalPause');
const intervalResetBtn = document.getElementById('intervalReset');

let intervalState = {
  intervals: [], // {name, time, color}
  idx: 0,
  remaining: 0,
  running: false,
  timerId: null
};

function buildIntervalsFromInputs() {
  const arr = [];
  const warm = Math.max(0, parseInt(warmupInput.value) || 0);
  const work = Math.max(0, parseInt(workInput.value) || 0);
  const rest = Math.max(0, parseInt(restInput.value) || 0);
  const cycles = Math.max(1, parseInt(cyclesInput.value) || 1);
  const cooldown = Math.max(0, parseInt(cooldownInput.value) || 0);

  if (warm > 0) arr.push({ name: 'Warmup', time: warm, color: 'ready' });
  for (let i = 1; i <= cycles; i++) {
    if (work > 0) arr.push({ name: `Work - ${i}`, time: work, color: 'work' });
    if (rest > 0 && i < cycles) arr.push({ name: `Rest - ${i}`, time: rest, color: 'rest' });
  }
  if (cooldown > 0) arr.push({ name: 'Cooldown', time: cooldown, color: 'rest' });
  return arr;
}

function setIntervalDisplay() {
  intervalTimeEl.textContent = formatSeconds(intervalState.remaining);
  // update progress inner
  const totalOfCurrent = intervalState.intervals[intervalState.idx]?.time || 1;
  const done = totalOfCurrent - intervalState.remaining;
  const percent = Math.round((done / totalOfCurrent) * 100);
  intervalProgressEl.style.width = `${percent}%`;
  intervalStatusEl.textContent = intervalState.intervals[intervalState.idx]?.name || 'Ready';
}

function startInterval() {
  if (intervalState.running) return;
  // if no intervals built or finished, build afresh
  if (!intervalState.intervals.length || intervalState.idx >= intervalState.intervals.length) {
    intervalState.intervals = buildIntervalsFromInputs();
    intervalState.idx = 0;
    if (!intervalState.intervals.length) {
      alert('Please set interval values first.');
      return;
    }
    intervalState.remaining = intervalState.intervals[0].time;
  }
  intervalState.running = true;
  intervalState.timerId = setInterval(tickInterval, 1000);
  updateIntervalButtons();
  setIntervalDisplay();
}

function tickInterval() {
  if (!intervalState.running) return;
  if (intervalState.remaining > 0) {
    intervalState.remaining -= 1;
    setIntervalDisplay();
  } else {
    // move to next
    if (intervalState.idx < intervalState.intervals.length - 1) {
      intervalState.idx += 1;
      intervalState.remaining = intervalState.intervals[intervalState.idx].time;
      setIntervalDisplay();
      // small beep at transition
      playBeepShort();
    } else {
      // finished
      clearInterval(intervalState.timerId);
      intervalState.running = false;
      playBeepLong();
      intervalStatusEl.textContent = 'Finished!';
      intervalProgressEl.style.width = '100%';
      updateIntervalButtons();
    }
  }
}

function pauseInterval() {
  if (!intervalState.running) return;
  clearInterval(intervalState.timerId);
  intervalState.running = false;
  updateIntervalButtons();
}

function resetInterval() {
  clearInterval(intervalState.timerId);
  intervalState = { intervals: [], idx: 0, remaining: 0, running: false, timerId: null };
  intervalProgressEl.style.width = '0%';
  intervalTimeEl.textContent = '00:00';
  intervalStatusEl.textContent = 'Ready';
  updateIntervalButtons();
}

function updateIntervalButtons() {
  intervalStartBtn.disabled = intervalState.running;
  intervalPauseBtn.disabled = !intervalState.running;
  intervalResetBtn.disabled = false;
}

/* attach */
intervalStartBtn.addEventListener('click', startInterval);
intervalPauseBtn.addEventListener('click', pauseInterval);
intervalResetBtn.addEventListener('click', resetInterval);

/* -------------------------
   STOPWATCH
   ------------------------- */
const stopwatchTimeEl = document.getElementById('stopwatchTime');
const stopwatchStatusEl = document.getElementById('stopwatchStatus');
const swStartStopBtn = document.getElementById('stopwatchStartStop');
const swLapBtn = document.getElementById('stopwatchLap');
const swResetBtn = document.getElementById('stopwatchReset');
const lapListEl = document.getElementById('lapList');

let stopwatch = {
  startAt: 0,
  elapsed: 0,
  running: false,
  timerId: null,
  laps: []
};

function startStopwatch() {
  if (stopwatch.running) return;
  stopwatch.running = true;
  stopwatch.startAt = Date.now() - stopwatch.elapsed;
  stopwatch.timerId = setInterval(() => {
    stopwatch.elapsed = Date.now() - stopwatch.startAt;
    // display at 50-100ms rate
    updateStopwatchDisplay();
  }, 50);
  stopwatchStatusEl.textContent = 'Running...';
  updateStopwatchButtons();
}

function pauseStopwatch() {
  if (!stopwatch.running) return;
  clearInterval(stopwatch.timerId);
  stopwatch.running = false;
  stopwatchStatusEl.textContent = 'Paused';
  updateStopwatchButtons();
}

function resetStopwatch() {
  clearInterval(stopwatch.timerId);
  stopwatch = { startAt: 0, elapsed: 0, running: false, timerId: null, laps: [] };
  updateStopwatchDisplay();
  lapListEl.innerHTML = '';
  stopwatchStatusEl.textContent = 'Ready';
  updateStopwatchButtons();
}

function recordLap() {
  if (!stopwatch.running) return;
  stopwatch.laps.unshift(stopwatch.elapsed);
  renderLaps();
}

function updateStopwatchDisplay() {
  stopwatchTimeEl.textContent = formatMs(stopwatch.elapsed);
}

function updateStopwatchButtons() {
  if (stopwatch.running) {
    swStartStopBtn.textContent = 'Pause';
    swStartStopBtn.classList.remove('bg-green-500'); swStartStopBtn.classList.add('bg-red-500');
    swLapBtn.disabled = false;
  } else {
    swStartStopBtn.textContent = stopwatch.elapsed > 0 ? 'Start' : 'Start';
    swStartStopBtn.classList.remove('bg-red-500'); swStartStopBtn.classList.add('bg-green-500');
    swLapBtn.disabled = stopwatch.elapsed === 0;
  }
}

function renderLaps() {
  if (!stopwatch.laps.length) {
    lapListEl.innerHTML = '<div class="text-sm text-gray-500">No laps</div>';
    return;
  }
  lapListEl.innerHTML = stopwatch.laps.map((t, i) => {
    const lapNum = stopwatch.laps.length - i;
    return `<div><span>Lap ${lapNum}</span><span>${formatMs(t)}</span></div>`;
  }).join('');
}

/* attach */
swStartStopBtn.addEventListener('click', () => {
  if (stopwatch.running) pauseStopwatch(); else startStopwatch();
});
swLapBtn.addEventListener('click', recordLap);
swResetBtn.addEventListener('click', resetStopwatch);

/* -------------------------
   COUNTDOWN
   ------------------------- */
const countdownTimeEl = document.getElementById('countdownTime');
const countdownProgressEl = document.getElementById('countdownProgress');
const countdownStatusEl = document.getElementById('countdownStatus');
const countdownStartStopBtn = document.getElementById('countdownStartStop');
const countdownResetBtn = document.getElementById('countdownReset');
const countdownMin = document.getElementById('countdownMin');
const countdownSec = document.getElementById('countdownSec');

let countdown = {
  total: 300,
  remaining: 300,
  running: false,
  timerId: null
};

function setCountdownFromInputs() {
  const m = Math.max(0, parseInt(countdownMin.value) || 0);
  const s = Math.max(0, parseInt(countdownSec.value) || 0);
  countdown.total = (m * 60) + s;
  countdown.remaining = countdown.total;
  updateCountdownDisplay();
}

function updateCountdownDisplay() {
  countdownTimeEl.textContent = formatSeconds(countdown.remaining);
  const percent = countdown.total ? Math.round(((countdown.total - countdown.remaining) / countdown.total) * 100) : 0;
  countdownProgressEl.style.width = `${percent}%`;
  countdownStatusEl.textContent = countdown.running ? 'Running' : (countdown.remaining === 0 ? "Time's up!" : 'Set Time');
}

function startCountdown() {
  if (countdown.running) return;
  // set from inputs if not started
  if (!countdown.total || countdown.remaining === 0) setCountdownFromInputs();
  if (countdown.total === 0) { alert('Set a positive time'); return; }
  countdown.running = true;
  countdown.timerId = setInterval(() => {
    if (countdown.remaining > 0) {
      countdown.remaining -= 1;
      updateCountdownDisplay();
    } else {
      clearInterval(countdown.timerId);
      countdown.running = false;
      playBeepLong();
      if (navigator.vibrate) navigator.vibrate(300);
      updateCountdownDisplay();
    }
  }, 1000);
  updateCountdownDisplay();
}

function pauseCountdown() {
  if (!countdown.running) return;
  clearInterval(countdown.timerId);
  countdown.running = false;
  updateCountdownDisplay();
}

function resetCountdown() {
  clearInterval(countdown.timerId);
  countdown.running = false;
  countdown.remaining = countdown.total || 0;
  updateCountdownDisplay();
}

/* attach */
countdownStartStopBtn.addEventListener('click', () => {
  if (countdown.running) pauseCountdown(); else startCountdown();
});
countdownResetBtn.addEventListener('click', resetCountdown);

/* -------------------------
   Sound helpers
   ------------------------- */
function playBeepShort() {
  if (!SOUND_ENABLED) return;
  try {
    beep.currentTime = 0;
    beep.play();
    if (navigator.vibrate) navigator.vibrate(120);
  } catch (e) { console.warn('beep fail', e); }
}
function playBeepLong() {
  if (!SOUND_ENABLED) return;
  try {
    // longer beep: play twice quickly
    beep.currentTime = 0;
    beep.play();
    setTimeout(() => { beep.currentTime = 0; beep.play().catch(()=>{}); }, 350);
    if (navigator.vibrate) navigator.vibrate([200, 80, 200]);
  } catch (e) { console.warn('beep fail', e); }
}

/* -------------------------
   Utilities: formatting
   ------------------------- */
function formatSeconds(s) {
  s = Math.max(0, Math.floor(s));
  const m = Math.floor(s / 60);
  const sec = s % 60;
  return `${String(m).padStart(2,'0')}:${String(sec).padStart(2,'0')}`;
}
function formatMs(ms) {
  ms = Math.max(0, Math.floor(ms));
  const totalSec = Math.floor(ms / 1000);
  const cs = Math.floor((ms % 1000) / 10);
  const m = Math.floor(totalSec / 60);
  const s = totalSec % 60;
  return `${String(m).padStart(2,'0')}:${String(s).padStart(2,'0')}.${String(cs).padStart(2,'0')}`;
}

/* -------------------------
   Startup & init
   ------------------------- */
function init() {
  // theme
  initTheme();

  // set initial countdown display
  setCountdownFromInputs();

  // show interval section by default
  showSection('interval');

  // bind initial lucide icons (already called in HTML, but ensure again)
  if (typeof lucide !== 'undefined' && lucide.createIcons) lucide.createIcons();

  // ensure sound toggle label is correct
  updateSoundToggleUI();

  // default lap list
  lapListEl.innerHTML = '<div class="text-sm text-gray-500">No laps</div>';
}
init();
